/'
	---------------------------------------------------------------------
	Copyright (C) 2017 Frank Hoogerbeets <contact@basicstudio.org>
	---------------------------------------------------------------------
	This library is free software; you can redistribute it and/or modify
	it under the terms of the Modified GNU Library General Public
	License either version 2.0 of the License, or (at your option) any
	later version.

	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

	See the Modified GNU Library General Public License for more details:
	http://www.basicstudio.org/mlgpl.html
	http://www.basicstudio.org/mlgpl.txt
	---------------------------------------------------------------------
'/

#include once "utf8.inc"

#define byte1_is_utf8 (((*p)[1] and &hC0) = &h80)
#define byte2_is_utf8 (((*p)[2] and &hC0) = &h80)
#define byte3_is_utf8 (((*p)[3] and &hC0) = &h80)

' house-keeping procedures
declare function UTF8Asc(p as pchar) as uinteger
declare function UTF8Cod(byval value as uinteger) as ustring
declare function UTF8CpLenX(p as pchar) as uinteger
declare function UTF8CpLen(p as pchar) as uinteger
declare function UTF8CpStart(p as pchar, byval length as uinteger, _
	byval index as uinteger) as pchar
declare function UTF8Eval(p as pchar, byval count as uinteger, _
	byval flag as boolean = true) as integer
declare function UTF8InstrP(sp as pchar, byval splen as uinteger, _
	sb as pchar, byval sblen as uinteger) as pchar
declare function UTF8Len(p as pchar, byval bytes as uinteger) _
	as uinteger
declare sub UTF8Repair(p as pchar)
declare sub UTF8ResizeBuffer (byref s as ustring, byref outp as pchar, _
	byval icnt as uinteger, byval ocnt as uinteger, _
	byval osize as integer, byval nsize as integer)
declare function UTF8Reverse(p as pchar, _
	byval count as uinteger) as ustring

' --- general procedures ---

function ByteIndex(b as any ptr, byval size as uinteger, _
byval chval as ubyte) as integer
	' returns -1 if byte was not found
	dim p as any ptr
	dim result as integer = -1
	p = memchr(b, chval, size)
	if p > 0 then
		result = p - b
	end if
	return result
end function

sub ResizeStr(byref s as string, byval size as uinteger)
	var slen = len(s)
	if size > slen then
		s += space(size - slen)
	elseif size < slen then
		s = left(s, size)
	end if
end sub

' string procedures supporting UTF-8

' UTF-8 value from unicode codepoint
function UAsc(byref value as const ustring) as uinteger
	' generates error if codepoint is illegal
	dim result as uinteger
	result = UTF8Asc(value)
	if result = 0 andalso (value <> chr(0)) then
		' illegal codepoint
		error 1 ' illegal function call
	end if
	return result
end function

' UTF-8 codepoint from unicode value
function UChr(byval value as uinteger) as ustring
	' generates error if codepoint is illegal
	dim result as ustring
	result = UTF8Cod(value)
	if result = "" then
		' illegal codepoint
		error 1 'illegal function call
	end if
	return result
end function

' UTF-8 insert string
function UInsert overload (byref s1 as const ustring, byref s2 _
as const ustring, byval start as uinteger) as const ustring
	return ULeft(s1, start - 1) + s2 + UMid(s1, start)
end function

' UTF-8 insert string with overwrite, remove
function UInsert overload (byref s1 as const ustring, byref s2 _
as const ustring, byval start as uinteger, byval count as uinteger) _
as const ustring
	return ULeft(s1, start - 1) + s2 + UMid(s1, start + count)
end function

' UTF-8 Instr (fixed start position)
function UInstr overload (byref sp as const ustring, _
byref sb as const ustring) as integer
	' with default start position
	return UInstr(1, sp, sb)
end function

' UTF-8 Instr (custom start position)
function UInstr overload (byval spos as uinteger, _
byref sp as const ustring, byref sb as const ustring) as integer
	' with custom start position
	dim as uinteger i, splen
	dim as pchar p, sposp
	if spos = 1 then
		i = instr(sp, sb)
		if i > 0 then
			return UTF8Len(sp, i - 1) + 1
		end if
	elseif spos > 1 then
		splen = len(sp)
		sposp = UTF8CpStart(sp, splen, spos - 1)
		if sposp = nil then
			return 0
		end if
		p = UTF8InstrP(sposp, splen + (strptr(sp) - sposp), sb, len(sb))
		if p = nil then
			return 0
		end if
		return spos + UTF8Len(sposp, p - sposp)
	end if
	return 0
end function

' UTF-8 is ascii
function UIsAscii(byref s as const ustring) as boolean
	' returns TRUE if string only contains codepoints < &h80
	for i as uinteger = 0 to len(s)
		if s[i] > &h7F then
			return false
		end if
	next
	return true
end function

' UTF-8 is valid string
function UIvstr(byref s as const string) as boolean
	return UTF8Eval(s, len(s)) = -1
end function

' UTF-8 left part of string
function ULeft(byref s as const ustring, byval count as uinteger) _
as const ustring
	return UMid(s, 1, count)
end function

' UTF-8 length
function ULen(byref s as const ustring) as uinteger
	return UTF8Len(s, len(s))
end function

' UTF-8 mid rest of the string
function UMid overload (byref s as const ustring, _
byval start as uinteger) as const ustring
	return UMid(s, start, ulen(s) - start + 1)
end function

' UTF-8 mid default
function UMid overload (byref s as const ustring, byval start _
as uinteger, byval count as uinteger) as const ustring
	dim as pchar sbpos, ebpos
	dim as uinteger maxb, slen
	dim result as ustring
	slen = len(s)
	sbpos = UTF8CpStart(s, slen, start - 1)
	result = ""
	if sbpos = nil then
		return result
	else
		maxb = strptr(s) + slen - sbpos
		ebpos = UTF8CpStart(sbpos, maxb, count)
		if ebpos = nil then
			result = mid(s, sbpos - strptr(s) + 1, maxb)
		else
			result = mid(s, sbpos - strptr(s) + 1, ebpos - sbpos)
		end if
	end if
	return result
end function

function URemove(byref s as ustring, byval start as uinteger, _
byval count as uinteger) as const ustring
	return UInsert(s, "", start, count)
end function

' UTF-8 repair string
sub URepair(byref s as string)
	' try to repair if not valid
	if not UIvstr(s) then
		UTF8Repair(s)
	end if
end sub

sub UReplace(byref t as ustring, byref i as const ustring, _
byref s as const ustring, byval a as integer = 1)
  dim as uinteger li, p
  p = instr(a, t, i)
  if p > 0 then
		li = len(i)
    if li <> len(s) then
			t = left(t, p - 1) + s + mid(t, p + li)
		else
			mid(t, p) = s
		end if
	end if
end sub

sub UReplaceAll(byref t as ustring, byref i as const ustring, _
byref s as const ustring, byval a as integer = 1)
  dim as uinteger li, ls, p
  p = instr(a, t, i)
  if p = 0 then
		exit sub
	end if
  li = len(i)
  ls = len(s)
  if li = ls then
		li = 0
	end if
  while p > 0
		if li then
			t = left(t, p - 1) + s + mid(t, p + li)
		else
			mid(t, p) = s
		end if
    p = instr(p + ls, t, i)
  wend
end sub

function UReverse(byref s as const ustring) as ustring
	dim as uinteger slen = len(s)
	if slen <> 0 then
		return UTF8Reverse(s, slen)
	end if
end function

function URight(byref s as const ustring, byval count as uinteger) _
as const ustring
	return UMid(s, ulen(s) - count + 1)
end function

function ULCase(byref s as const ustring, byref lang _
as const string = "") as const ustring
	dim as uinteger cdiff, p
	dim as pcchar istr, istrend
	dim as pchar outp
	dim as boolean IsTurkish
	dim as ubyte c1, c2, c3, nc1, nc2, nc3
	dim as ustring result

	if len(s) = 0 then
		exit function
	end if

	istr = strptr(s)
	istrend = istr + len(s)
	result = s

	while istr < istrend
		c1 = (*istr)[0]
		select case c1
			case &h41 to &h5A
				exit while
			case &hC3 to &hFF
				select case c1
					case &hC3 to &hC9, &hCE, &hCF, &hD0 to &hD5, &hE1, &hE2, &hE5
						c2 = (*istr)[1]
						select case c1
							case &hC3
								if within(&h80, &h9E, c2) then
									exit while
								end if
							case &hC4
								select case c2
									case &h80 to &hAF, &hB2 to &hB6
										if (c2 mod 2) = 0 then
											exit while
										end if
									case &hB8 to &hFF
										if (c2 mod 2) = 1 then
											exit while
										end if
									case &hB0
										exit while
								end select
							case &hC5
								select case c2
									case &h8A to &hB7
										if (c2 mod 2) = 0 then
											exit while
										end if
									case &h00 to &h88, &hB9 to &hFF
										if (c2 mod 2) = 1 then
											exit while
										end if
									case &hB8
										exit while
								end select
							case &hE5
								if c2 = &hBC then
									if within(&hA1, &hBA, (*istr)[2]) then
										exit while
									end if
								end if
							case else
								exit while
						end select
				end select
		end select
    istr += 1
	wend

	if istr >= istrend then
		return result
	end if

	if lang = "tr" orelse lang = "az" then
		IsTurkish = true
	end if

	outp = strptr(result) + (istr - strptr(s))
	cdiff = 0

  while istr < istrend
		c1 = (*istr)[0]
		select case c1
			case &h41 to &h5A
				' ASCII
				if IsTurkish andalso (c1 = asc("I")) then
					p = outp - strptr(result)
					ResizeStr(result, len(result) + 1)
					outp = strptr(result) + p
					(*outp)[0] = &hC4
					outp += 1
					(*outp)[0] = &hB1
					cdiff -= 1
				else
					(*outp)[0] = c1 + &h20
				end if
				istr += 1
				outp += 1
			case &hC3 to &hD5
				' two bytes
				c2 = (*istr)[1]
				nc1 = c1
				nc2 = c2
				select case c1
					case &hC3
						select case c2
							case &h80 to &h96, &h98 to &h9E
								nc2 = c2 + &h20
						end select
					case &hC4
						select case c2
							case &h80 to &hAF, &hB2 to &hB7
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &hB0
								(*outp)[0] = asc("i")
								istr += 2
								outp += 1
								cdiff += 1
								continue while
							case &hB9 to &hBE
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &hBF
								nc1 = &hC5
								nc2 = &h80
						end select
					case &hC5
						select case c2
							case &h8A to &hB7
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &h00 to &h88, &hB9 to &hBE
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &hB8
								nc1 = &hC3
								nc2 = &hBF
						end select
					case &hC6
						select case c2
							case &h81
								nc1 = &hC9
								nc2 = &h93
							case &h82 to &h85
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &h87, &h88, &h8B, &h8C
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &h86
								nc1 = &hC9
								nc2 = &h94
							case &h89
								nc1 = &hC9
								nc2 = &h96
							case &h8A
								nc1 = &hC9
								nc2 = &h97
							case &h8E
								nc1 = &hC7
								nc2 = &h9D
							case &h8F
								nc1 = &hC9
								nc2 = &h99
							case &h90
								nc1 = &hC9
								nc2 = &h9B
							case &h91, &h98
								nc2 = c2 + 1
							case &h93
								nc1 = &hC9
								nc2 = &hA0
							case &h94
								nc1 = &hC9
								nc2 = &hA3
							case &h96
								nc1 = &hC9
								nc2 = &hA9
							case &h97
								nc1 = &hC9
								nc2 = &hA8
							case &h9C
								nc1 = &hC9
								nc2 = &hAF
							case &h9D
								nc1 = &hC9
								nc2 = &hB2
							case &h9F
								nc1 = &hC9
								nc2 = &hB5
							case &hA0 to &hA5, &hAC
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &hA7, &hAF
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &hA6
								nc1 = &hCA
								nc2 = &h80
							case &hA9
								nc1 = &hCA
								nc2 = &h83
							case &hAE
								nc1 = &hCA
								nc2 = &h88
							case &hB8, &hBC
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &hB3 to &hB6
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &hB1
								nc1 = &hCA
								nc2 = &h8A
							case &hB2
								nc1 = &hCA
								nc2 = &h8B
							case &hB7
								nc1 = &hCA
								nc2 = &h92
						end select
					case &hC7
						select case c2
							case &h84 to &h8C, &hB1 to &hB3
								if ((c2 and &hF) mod 3) = 1 then
									nc2 = c2 + 2
								elseif ((c2 and &hF) mod 3) = 2 then
									nc2 = c2 + 1
								end if
							case &h8D to &h9C
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &h9E to &hAF, &hB4, &hB5, &hB8 to &hBF
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &hB6
								nc1 = &hC6
								nc2 = &h95
							case &hB7
								nc1 = &hC6
								nc2 = &hBF
						end select
					case &hC8
						if within(&h80, &hB3, c2) then
							if (c2 mod 2) = 0 then
								nc2 = c2 + 1
							end if
						end if
						select case c2
							case &hA0
								nc1 = &hC6
								nc2 = &h9E
							case &hA1
								nc2 = c2
							case &hBA, &hBE
								p = outp - strptr(result)
								ResizeStr(result, len(result) + 1)
								outp = strptr(result) + p
								(*outp)[0] = &hE2
								outp += 1
								(*outp)[0] = &hB1
								outp += 1
								if c2 = &hBA then
									(*outp)[0] = &hA5
								else
									(*outp)[0] = &hA6
								end if
								cdiff -= 1
								outp += 1
								istr += 2
								continue while
							case &hBD
								nc1 = &hC6
								nc2 = &h9A
							case &hBB
								nc2 = c2 + 1
						end select
					case &hC9
						select case c2
							case &h81, &h82
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &h86 to &h8F
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &h83
								nc1 = &hC6
								nc2 = &h80
							case &h84
								nc1 = &hCA
								nc2 = &h89
							case &h85
								nc1 = &hCA
								nc2 = &h8C
						end select
					case &hCE
						select case c2
							case &h86
								nc2 = &hAC
							case &h88
								nc2 = &hAD
							case &h89
								nc2 = &hAE
							case &h8A
								nc2 = &hAF
							case &h8C
								' nc2 doesn't change
								nc1 = &hCF
							case &h8E
								nc1 = &hCF
								nc2 = &h8D
							case &h8F
								nc1 = &hCF
								nc2 = &h8E
							case &h91 to &h9F
								nc2 = c2 + &h20
							case &hA0 to &hAB
								nc1 = &hCF
								nc2 = c2 - &h20
						end select
					case &hCF
						select case c2
							case &h8F
								nc2 = &h97
							case &h98
								nc2 = &h99
							case &h9A
								nc2 = &h9B
							case &h9C
								nc2 = &h9D
							case &h9E
								nc2 = &h9F
							case &hA0 to &hAF
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
							case &hB4
								nc1 = &hCE
								nc2 = &hB8
							case &hB7
								nc2 = &hB8
							case &hB9
								nc2 = &hB2
							case &hBA
								nc2 = &hBB
							case &hBD
								nc1 = &hCD
								nc2 = &hBB
							case &hBE
								nc1 = &hCD
								nc2 = &hBC
							case &hBF
								nc1 = &hCD
								nc2 = &hBD
						end select
					case &hD0
						c2 = (*istr)[1]
						select case c2
							case &h80 to &h8F
								nc1 = c1 + 1
								nc2 = c2 + &h10
							case &h90 to &h9F
								nc2 = c2 + &h20
							case &hA0 to &hAF
								nc1 = c1 + 1
								nc2 = c2 - &h20
						end select
					case &hD1
						if within(&hA0, &hBF, c2) then
							if (c2 mod 2) = 0 then
								nc2 = c2 + 1
							end if
						end if
					case &hD2
						select case c2
							case &h80
								nc2 = c2 + 1
							case &h8A to &hBF
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
						end select
					case &hD3
						select case c2
							case &h80
								nc2 = &h8F
							case &h81 to &h8E
								if (c2 mod 2) = 1 then
									nc2 = c2 + 1
								end if
							case &h90 to &hBF
								if (c2 mod 2) = 0 then
									nc2 = c2 + 1
								end if
						end select
					case &hD4
						if (c2 mod 2) = 0 then
							nc2 = c2 + 1
						end if
						if within(&hB1, &hBF, c2) then
							nc1 = &hD5
							nc2 = c2 - &h10
						end if
					case &hD5
						select case c2
							case &h80 to &h8F
								nc2 = c2 + &h30
							case &h90 to &h96
								nc1 = &hD6
								nc2 = c2 - &h10
						end select
				end select
				' evaluate
				if cdiff <> 0 then
					(*outp)[0] = nc1
					(*outp)[1] = nc2
				else
					if nc1 <> c1 then (*outp)[0] = nc1
					if nc2 <> c2 then (*outp)[1] = nc2
				end if
				istr += 2
				outp += 2
			' three bytes
			case &hE1
				nc1 = c1
				c2 = (*istr)[1]
				c3 = (*istr)[2]
				nc2 = c2
				nc3 = c3
				select case c2
					case &h82
						if within(&hA0, &hBF, c3) then
								nc1 = &hE2
								nc2 = &hB4
								nc3 = c3 - &h20
						end if
					case &h83
						if within(&h80, &h85, c3) then
								nc1 = &hE2
								nc2 = &hB4
								nc3 = c3 + &h20
						end if
					case &hB8 to &hBB
						if (c3 mod 2) = 0 then
							nc3 = c3 + 1
						end if
						if c2 = &hBA then
							if within(&h96, &h9F, c3) then
								nc3 = c3
							end if
						end if
						if c2 = &hBA andalso c3 = &h9E then
							istr += 3
							(*outp)[0] = &hC3
							outp += 1
							(*outp)[0] = &h9F
							outp += 1
							cdiff += 1
							continue while
						end if
					case &hBC
						if (c3 mod &h10) \ 8 = 1 then
							nc3 = c3 - 8
						end if
					case &hBD
						select case c3
							case &h80 to &h8F, &hA0 to &hAF
								if (c3 mod &h10) \ 8 = 1 then
									nc3 = c3 - 8
								end if
							case &h99, &h9B, &h9D, &h9F
								nc3 = c3 - 8
						end select
					case &hBE
						select case c3
							case &h80 to &hB9
								if (c3 mod &h10) \ 8 = 1 then
									nc3 = c3 - 8
								end if
							case &hBA
								nc2 = &hBD
								nc3 = &hB0
							case &hBB
								nc2 = &hBD
								nc3 = &hB1
							case &hBC
								nc3 = &hB3
						end select
				end select
				if cdiff <> 0 then
					(*outp)[0] = nc1
					(*outp)[1] = nc2
					(*outp)[2] = nc3
				else
					if c1 <> nc1 then (*outp)[0] = nc1
					if c2 <> nc2 then (*outp)[1] = nc2
					if c3 <> nc3 then (*outp)[2] = nc3
				end if
				istr += 3
				outp += 3
			case &hE2
				nc1 = c1
				c2 = (*istr)[1]
				c3 = (*istr)[2]
				nc2 = c2
				nc3 = c3
				select case c2
					case &h84
						select case c3
							case &hA6
								istr += 3
								(*outp)[0] = &hCF
								outp += 1
								(*outp)[0] = &h89
								outp += 1
								cdiff += 1
								continue while
							case &hAA
								istr += 3
								(*outp)[0] = &h6B
								outp += 1
								cdiff += 2
								continue while
							case &hAB
								istr += 3
								(*outp)[0] = &hC3
								outp += 1
								(*outp)[0] = &hA5
								outp += 1
								cdiff += 1
								continue while
						end select
					case &h85
						if within(&hA0, &hAF, c3) then
							nc3 = c3 + &h10
						end if
					case &h86
						if c3 = &h83 then
							nc3 = c3 + 1
						end if
					case &h92
						if within(&hB6, &hBF, c3) then
							nc2 = &h93
							nc3 = c3 - &h26
						end if
					case &h93
						if within(&h80, &h8F, c3) then
							nc3 = c3 + &h26
						end if
					case &hB0
						select case c3
							case &h80 to &h8F
								nc3 = c3 + &h30
							case &h90 to &hAE
								nc2 = &hB1
								nc3 = c3 - &h10
						end select
					case &hB1
						select case c3
							case &hA0
								nc3 = c3 + 1
							case &hA2, &hA4, &hAD to &hAF, &hB0
								istr += 3
								(*outp)[0] = &hC9
								outp += 1
								select case c3
									case &hA2
										(*outp)[0] = &hAB
									case &hA4
										(*outp)[0] = &hBD
									case &hAD
										(*outp)[0] = &h91
									case &hAE
										(*outp)[0] = &hB1
									case &hAF
										(*outp)[0] = &h90
									case &hB0
										(*outp)[0] = &h92
								end select
								outp += 1
								cdiff += 1
								continue while
							case &hA3
								nc2 = &hB5
								nc3 = &hBD
							case &hA7, &hA9, &hAB
								nc3 = c3 + 1
							case &hB2, &hB5
								nc3 = c3 + 1
							case &hBE, &hBF
								istr += 3
								(*outp)[0] = &hC8
								outp += 1
								if c3 = &hBE then
									(*outp)[0] = &hBF
								else
									(*outp)[0] = &h80
								end if
								outp += 1
								cdiff += 1
								continue while
						end select
					case &hB2
						if (c3 mod 2) = 0 then
							nc3 = c3 + 1
						end if
					case &hB3
						if within(&h80, &hA3, c3) then
							if (c3 mod 2) = 0 then nc3 = c3 + 1
						end if
				end select
				if cdiff <> 0 then
					(*outp)[0] = nc1
					(*outp)[1] = nc2
					(*outp)[2] = nc3
				else
					if c1 <> nc1 then (*outp)[0] = nc1
					if c2 <> nc2 then (*outp)[1] = nc2
					if c3 <> nc3 then (*outp)[2] = nc3
				end if
				istr += 3
				outp += 3
			case &hEF
				c2 = (*istr)[1]
				c3 = (*istr)[2]
				if c2 = &hBC then
					if within(&hA1, &hBA, c3) then
						(*outp)[0] = c1
						(*outp)[1] = &hBD
						(*outp)[2] = c3 - &h20
					end if
				end if
				if cdiff <> 0 then
					(*outp)[0] = c1
					(*outp)[1] = c2
					(*outp)[2] = c3
				end if
				istr += 3
				outp += 3
			case else
				if cdiff <> 0 then (*outp)[0] = c1
				istr += 1
				outp += 1
		end select
  wend
  ' set final buffer size
	ResizeStr(result, outp - strptr(result))
	return result
end function

function UUCase(byref s as const ustring, byref lang _
as const string = "") as const ustring
	dim as uinteger icnt, ocnt
	dim as pchar outp
	dim as ubyte cplen, ncplen
	dim as ushort ncod, ocod
	dim as boolean codpr, IsTurkish
	dim as ustring result

	if len(s) = 0 then
		exit function
	end if

	result = s
	outp = strptr(result)
	if lang = "tr" orelse lang = "az" then
		IsTurkish = true
	end if

	icnt = 0
	ocnt = 0

	while icnt <= len(s)
		if within(&h61, &h7A, s[icnt]) then
			if IsTurkish andalso (s[icnt] = asc("i")) then
				ResizeStr(result, len(result) + 1)
				outp = strptr(result)
				(*outp)[ocnt] = &hC4
				(*outp)[ocnt + 1] = &hB0
				icnt += 1
				ocnt += 2
			else
				(*outp)[ocnt] = s[icnt] - &h20
				icnt += 1
				ocnt += 1
			end if
		else
			cplen = UTF8CpLen(cast(pchar, @s[icnt]))
			codpr = false
			ncplen = cplen
			if cplen = 2 then
				ocod = (s[icnt] shl 8) or s[icnt + 1]
				ncod = 0
				select case ocod
					case &hC39F
						ncod = &h5353
					case &hC3A0 to &hC3B6, &hC3B8 to &hC3BE
						ncod = ocod - &h20
					case &hC3BF
						ncod = &hC5B8
					case &hC481 to &hC4B0
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hCB1
						(*outp)[ocnt] = asc("I")
						ncplen = 1
						codpr = true
					case &hC4B2 to &hC4B7
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hC4B9 to &hC4BF
						if (ocod mod 2) = 0 then
							ncod = ocod - 1
						end if
					case &hC580
						ncod = &hC4BF
					case &hC581 to &hC588
						if (ocod mod 2) = 0 then
							ncod = ocod - 1
						end if
					case &hC58A to &hC5B7
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hC5B9 to &hC5BE
						if (ocod mod 2) = 0 then
							ncod = ocod - 1
						end if
					case &hC5BF
						(*outp)[ocnt] = asc("S")
						ncplen = 1
						codpr = true
					case &hC680
						ncod = &hC983
					case &hC682 to &hC685
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hC688
						ncod = &hC687
					case &hC68C
						ncod = &hC68B
					case &hC692
						ncod = &hC691
					case &hC695
						ncod = &hC7B6
					case &hC699
						ncod = &hC698
					case &hC69A
						ncod = &hC8BD
					case &hC69E
						ncod = &hC8A0
					case &hC6A0 to &hC6A5
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hC6A8
						ncod = &hC6A7
					case &hC6AD
						ncod = &hC6AC
					case &hC6B0
						ncod = &hC6AF
					case &hC6B3 to &hC6B6
						if (ocod mod 2) = 0 then
							ncod = ocod - 1
						end if
					case &hC6B9
						ncod = &hC6B8
					case &hC6BD
						ncod = &hC6BC
					case &hC6BF
						ncod = &hC7B7
					case &hC784 to &hC786
						ncod = &hC784
					case &hC787 to &hC789
						ncod = &hC787
					case &hC78A to &hC78C
						ncod = &hC78A
					case &hC78E
						ncod = &hC78D
					case &hC790
						ncod = &hC78F
					case &hC791 to &hC79C
						if (ocod mod 2) = 0 then
							ncod = ocod - 1
						end if
					case &hC79D
						ncod = &hC68E
					case &hC79F
						ncod = &hC79E
					case &hC7A0 to &hC7AF
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hC7B2 to &hC7B3
						ncod = &hC7B1
					case &hC7B5
						ncod = &hC7B4
					case &hC7B8 to &hC7BF
						if ocod mod 2 = 1 then
							ncod = ocod - 1
						end if
					case &hC880 to &hC89F
						if ocod mod 2 = 1 then
							ncod = ocod - 1
						end if
					case &hC8A2 to &hC8B3
						if ocod mod 2 = 1 then
							ncod = ocod - 1
						end if
					case &hC8BC
						ncod = &hC8BB
					case &hC8BF
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hBE
						ncplen = 3
						codpr = true
					case &hC980
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hBF
						ncplen = 3
						codpr = true
					case &hC982
						ncod = &hC981
					case &hC986 to &hC98F
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hC990
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hAF
						ncplen = 3
						codpr = true
					case &hC991
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hAD
						ncplen = 3
						codpr = true
					case &hC992
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hB0
						ncplen = 3
						codpr = true
					case &h993
						ncod = &hC681
					case &hC994
						ncod = &hC686
					case &hC996
						ncod = &hC689
					case &hC997
						ncod = &hC68A
					case &hC999
						ncod = &hC68F
					case &hC99B
						ncod = &hC690
					case &hC9A0
						ncod = &hC693
					case &hC9A3
						ncod = &hC694
					case &hC9A5
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hEA
						(*outp)[ocnt + 1] = &h9E
						(*outp)[ocnt + 2] = &h8D
						ncplen = 3
						codpr = true
					case &hC9A8
						ncod = &hC697
					case &hC9A9
						ncod = &hC696
					case &hC9AB
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hA2
						ncplen = 3
						codpr = true
					case &hC9AF
						ncod = &hC69C
					case &hC9B1
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hAE
						ncplen = 3
						codpr = true
					case &hC9B2
						ncod = &hC69D
					case &hC9B5
						ncod = &hC69F
					case &hC9BD
						UTF8ResizeBuffer(result, outp, icnt, ocnt, 2, 3)
						(*outp)[ocnt] = &hE2
						(*outp)[ocnt + 1] = &hB1
						(*outp)[ocnt + 2] = &hA4
						ncplen = 3
						codpr = true
					case &hCA80
						ncod = &hC6A6
					case &hCA83
						ncod = &hC6A9
					case &hCA88
						ncod = &hC6AE
					case &hCA89
						ncod = &hC984
					case &hCA8A
						ncod = &hC6B1
					case &hCA8B
						ncod = &hC6B2
					case &hCA8C
						ncod = &hC985
					case &hCA92
						ncod = &hC6B7
					case &hCEAC
						ncod = &hCE86
					case &hCEAD
						ncod = &hCE88
					case &hCEAE
						ncod = &hCE89
					case &hCEAF
						ncod = &hCE8A
					case &hCEB1 to &hCEBF
						ncod = ocod - &h20
					case &hCF80, &hCF81, &hCF83 to &hCF8B
						ncod = ocod - &hE0
					case &hCF82
						ncod = &hCEA3
					case &hCF8C
						ncod = &hCE8C
					case &hCF8D
						ncod = &hCE8E
					case &hCF8E
						ncod = &hCE8F
					case &hCF90
						ncod = &hCE92
					case &hCF91
						ncod = &hCE98
					case &hCF95
						ncod = &hCEA6
					case &hCF96
						ncod = &hCEA0
					case &hCF97
						ncod = &hCF8F
					case &hCF99 to &hCF9F, &hCFA0 to &hCFAF
						if (ocod mod 2) = 1 then
							ncod = ocod - 1
						end if
					case &hCFB0
						ncod = &hCE9A
					case &hCFB1
						ncod = &hCEA1
					case &hCFB2
						ncod = &hCFB9
					case &hCFB5
						ncod = &hCE95
					case &hCFB8
						ncod = &hCFB7
					case &hCFBB
						ncod = &hCFBA
					case &hD0B0 to &hD0BF
						ncod = ocod - &h20
					case &hD180 to &hD18F
						ncod = ocod - &hE0
					case &hD190 to &hD19F
						ncod = ocod - &h110
				end select
				if ncod <> 0 then
					(*outp)[ocnt] = hibyte(ncod)
					(*outp)[ocnt + 1] = lobyte(ncod)
					codpr = true
				end if
			end if
			if (icnt <> ocnt + 1) andalso (codpr = false) then
				for i as uinteger = 0 to cplen - 1
					(*outp)[ocnt + i] = s[icnt + i]
				next
			end if
			icnt += cplen
			ocnt += ncplen
		end if
	wend
	' -1 prevents trailing space character (tested with ASCII and UTF8)
	ResizeStr(result, ocnt - 1)
	return result
end function

' --- house-keeping routines ---

function UTF8Asc(p as pchar) as uinteger
	dim result as uinteger
	if p <> nil then
		if (*p)[0] < &hC0 then
			' one byte
			result = (*p)[0]
		elseif ((*p)[0] and &hE0) = &hC0 then
			' two bytes?
			if byte1_is_utf8 then
				result = (((*p)[0] and &h1F) shl 6) _
					or ((*p)[1] and &h3F)
				if result < &h80 then
					' illegal encoding, perhaps XSS attack
					result = 0
				end if
			else
				' single character
				result = (*p)[0]
			end if
		elseif ((*p)[0] and &hF0) = &hE0 then
			' three bytes?
			if byte1_is_utf8 _
			andalso byte2_is_utf8 then
				result = (((*p)[0] and &h1F) shl 12) _
					or (((*p)[1] and &h3F) shl 6) _
					or ((*p)[2] and &h3F)
				if result < &h800 then
					' illegal encoding, perhaps XSS attack
					result = 0
				end if
			else
				' single character
				result = (*p)[0]
			end if
		elseif ((*p)[0] and &hF8) = &hF0 then
			' four bytes?
			if byte1_is_utf8 _
			andalso byte2_is_utf8 _
			andalso byte3_is_utf8 then
				result = (((*p)[0] and &hF) shl 18) _
					or (((*p)[1] and &h3F) shl 12) _
					or (((*p)[2] and &h3F) shl 6) _
					or ((*p)[3] and &h3F)
				if result < &h10000 then
					' illegal encoding, perhaps XSS attack
					result = 0
				end if
			else
				' single character
				result = (*p)[0]
			end if
		else
			' illegal character
			result = 0
		end if
	end if
	return result
end function

function UTF8CpLenX(p as pchar) as uinteger
	dim result as uinteger = 1
	select case (*p)[0]
		case &hC0 to &hDF
			' two bytes
			if byte1_is_utf8 then
				result = 2
			end if
		case &hE0 to &hEF
			' three bytes
			if byte1_is_utf8 andalso byte2_is_utf8 then
				result = 3
			end if
		case &hF0 to &hF7
			' four bytes
			if byte1_is_utf8 then
				if byte2_is_utf8 andalso byte3_is_utf8 then
					result = 4
				end if
			end if
	end select
	return result
end function

function UTF8CpLen(p as pchar) as uinteger
	if p = nil then
		return 0
	end if
	if (*p)[0] < &hC0 then
		return 1
	end if
	return UTF8CpLenX(p)
end function

function UTF8Cod(byval value as uinteger) as ustring
	' unicode value to codepoint string
	dim result as ustring
	select case value
		case &h0 to &h7F
			' one byte
			result = space(1)
			result[0] = value
		case &h80 to &h7FF
			' two bytes
			result = space(2)
			result[0] = &hC0 or (value shr 6)
			result[1] = &h80 or (value and &h3F)
		case &h800 to &hFFFF
			' three bytes
			result = space(3)
			result[0] = &hE0 or (value shr 12)
			result[1] = ((value shr 6) and &h3F) or &h80
			result[2] = (value and &h3F) or &h80
		case &h10000 to &h10FFFF
			' four bytes
			result = space(4)
			result[0] = &hF0 or (value shr 18)
			result[1] = ((value shr 12) and &h3F) or &h80
			result[2] = ((value shr 6) and &h3F) or &h80
			result[3] = (value and &h3F) or &h80
	end select
	return result
end function

function UTF8CpStart(p as pchar, byval length as uinteger, _
byval index as uinteger) as pchar
	' codepoint start
	dim cplen as uinteger
	dim result as pchar = p
	if result <> 0 then
		while (index > 0) andalso (length > 0)
			cplen = UTF8CpLen(result)
			length -= cplen
			index -= 1
			result += cplen
		wend
		if (index <> 0) orelse (length < 0) then
			result = nil
		end if
	end if
	return result
end function

function UTF8Eval(p as pchar, byval count as uinteger, _
byval flag as boolean = true) as integer
	dim cplen as integer
	dim c as ubyte
	dim result as integer
	if p <> nil then
		result = 0
		while result < count
			c = (*p)[0]
			if c < &h80 then
				cplen = 1
			elseif c <= &hC1 then
				' flag = true means exit if error
				if flag orelse (c >= &hC0) then
					return result
				end if
				cplen = 1
			elseif c <= &hDF then
				if (result < count - 1) then
					if byte1_is_utf8 then
						cplen = 2
					end if
				else
					return result
				end if
			elseif c <= &hEF then
				if (result < count - 2) then
					if byte1_is_utf8 andalso byte2_is_utf8 then
						if (c = &hE0) andalso ((*p)[1] <= &h9F) then
							return result
						end if
						cplen = 3
					end if
				else
					return result
				end if
			elseif c <= &hF7 then
				if (result < count - 3) then
					if byte1_is_utf8 _
					andalso byte2_is_utf8 _
					andalso byte3_is_utf8 then
						if (c = &hF0) andalso ((*p)[1] <= &h8F) then
							return result
						end if
						cplen = 4
					else
						return result
					end if
				end if
			else
				if flag then
					return result
				end if
				cplen = 1
			end if
			result += cplen
			p += cplen
			if result > count then
				result -= cplen
				return result
			end if
		wend
	end if
	return -1
end function

sub UTF8Repair(p as pchar)
	dim c as uinteger
	if p = nil then exit sub
	while (*p)[0] <> 0
		if (*p)[0] < &h80 then
			' single byte
			p += 1
		elseif (*p)[0] < &hC0 then
			' invalid
			(*p)[0] = 32
			p += 1
		elseif ((*p)[0] and &hE0) = &hC0 then
			' two bytes
			if byte1_is_utf8 then
				c = ((*p)[0] and &h1F) shl 6
				if c < &h80 then
					' fix XSS attack
					(*p)[0] = 32
				else
					p += 2
				end if
			else
				' invalid
				(*p)[0] = 32
				p += 1
			end if
		elseif ((*p)[0] and &hF0) = &hE0 then
			' three bytes
			if byte1_is_utf8 andalso byte2_is_utf8 then
				c = (((*p)[0] and &h1F) shl 12) _
					or (((*p)[1] and &h3F) shl 6)
				if c < &h800 then
					' fix XSS attack
					(*p)[0] = 32
				else
					p += 3
				end if
			else
				' invalid
				(*p)[0] = 32
				p += 1
			end if
		elseif ((*p)[0] and &hF8) = &hF0 then
			if byte1_is_utf8 _
			andalso byte2_is_utf8 _
			andalso byte3_is_utf8 then
				c = (((*p)[0] and &hF) shl 18) _
					or (((*p)[1] and &h3F) shl 12) _
					or (((*p)[2] and &h3F) shl 6)
				if c < &h10000 then
					' fix XSS attack
					(*p)[0] = 32
				else
					p += 4
				end if
			else
				' invalid
				(*p)[0] = 32
				p += 1
			end if
		else
			' invalid
			(*p)[0] = 32
			p += 1
		end if
	wend
end sub

function UTF8InstrP(sp as pchar, byval splen as uinteger, _
sb as pchar, byval sblen as uinteger) as pchar
	' returns pointer to search string, or 0 if not found
	dim p as integer
	if (sp = nil) orelse (sb = nil) orelse (sblen = 0) then
		return nil
	end if
	while splen > 0
		p = ByteIndex(sp, splen, (*sb)[0])
		if p < 0 then
			exit function
		end if
		sp += p
		splen -= p
		if splen < sblen then
			exit function
		end if
		if memcmp(sp, sb, sblen) = 0 then
			return sp
		end if
		sp += 1
		splen -= 1
	wend
	return 0
end function

function UTF8Len(p as pchar, byval bytes as uinteger) as uinteger
	dim cplen as uinteger
	dim result as uinteger = 0
	while bytes > 0
		result += 1
		cplen = UTF8CpLen(p)
		p += cplen
		bytes -= cplen
	wend
	return result
end function

sub UTF8ResizeBuffer (byref s as ustring, byref outp as pchar, _
byval icnt as uinteger, byval ocnt as uinteger, _
byval osize as integer, nsize as integer)
	if not (nsize > osize) then
		exit sub
	end if
	if (nsize > 20) orelse (osize > 20) then
		exit sub
	end if
	if (nsize > osize) andalso (ocnt >= icnt - 1) then
		ResizeStr(s, len(s) + nsize - osize)
		outp = strptr(s)
	end if
end sub

function UTF8Reverse(p as pchar, _
	byval count as uinteger) as ustring
	dim as uinteger rbpos
	dim as ubyte cplen
	dim as ustring result
	ResizeStr(result, count)
	rbpos = count
	while rbpos > 0
		cplen = UTF8CpLen(p)
		rbpos -= cplen
		memcpy(@result[rbpos], p, cplen)
		p += cplen
	wend
	return result
end function
