' ******************************************************************
'   Mersenne Twister Random Number Generator
' ******************************************************************
'
'   A C-program for MT19937, with initialization improved 2002/1/26. 
'   Coded by Takuji Nishimura and Makoto Matsumoto. 
'
'   Translated to FreeBASIC by Greg Lyon - April 2006 
'   Adapted for FBMath by Jean Debord - May 2006 / February 2007
'
'   Before using, initialize the state by using init_genrand(seed)
'   or init_by_array(init_key, key_length) (respectively InitMT and
'   InitMTbyArray in the FB version)
'
'   Copyright (C) 1997 - 2002, Makoto Matsumoto and Takuji Nishimura, 
'   All rights reserved.
' 
'   Redistribution and use in source and binary forms, with or without 
'   modification, are permitted provided that the following conditions 
'   are met: 
' 
'     1. Redistributions of source code must retain the above copyright 
'        notice, this list of conditions and the following disclaimer. 
' 
'     2. Redistributions in binary form must reproduce the above copyright 
'        notice, this list of conditions and the following disclaimer in the 
'        documentation and/or other materials provided with the distribution. 
' 
'     3. The names of its contributors may not be used to endorse or promote 
'        products derived from this software without specific prior written 
'        permission. 
'
'   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
'   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
'   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
'   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
'   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
'   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
'   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
'   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
'   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
'   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
'   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
'
'   Any feedback is very welcome.
'   http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
'   email: m-mat @ math.sci.hiroshima-u.ac.jp (remove space)
' ******************************************************************

CONST NN AS UINTEGER = 624
CONST MM AS UINTEGER = 397

CONST MATRIX_A   AS UINTEGER = &H9908b0dfUL  ' constant vector a
CONST UPPER_MASK AS UINTEGER = &H80000000UL  ' most significant w-r bits
CONST LOWER_MASK AS UINTEGER = &H7fffffffUL  ' least significant r bits

DIM SHARED mt(0 TO NN - 1) AS UINTEGER  ' the array For the state vector 
DIM SHARED mti AS UINTEGER = NN + 1     ' mti == NN+1 means mt[NN] is not initialized

DIM SHARED AS UINTEGER mag01(0 TO 1) = {&H0UL, MATRIX_A}

SUB InitMT(BYVAL Seed AS UINTEGER)
' Initializes mt with a seed

  DIM AS UINTEGER i

  mt(0)= Seed AND &HffffffffUL

  FOR i = 1 TO NN-1
    mt(i) = 1812433253UL * (mt(i-1) XOR (mt(i-1) SHR 30)) + i
    ' See Knuth TAOCP Vol2. 3rd Ed. P.106 For multiplier.
    ' In the previous versions, MSBs of the seed affect
    ' only MSBs of the array mt[].
    ' 2002/01/09 modified by Makoto Matsumoto
    mt(i) = mt(i) AND &HffffffffUL
    ' For >32 Bit machines
  NEXT i

  mti = NN
END SUB

SUB InitMTbyTime()
' Initialize by the value of TIMER

  InitMT(CUINT(TIMER * 1000))
END SUB

SUB InitMTbyArray(InitKey() AS UINTEGER)
' Initialize by an array InitKey(0 to (key_length - 1))

  DIM AS UINTEGER i, j, k, k1, key_length

  InitMT(19650218UL)

  i = 1
  j = 0

  key_length = UBOUND(InitKey) + 1
  IF NN > key_length THEN k1 = NN ELSE k1 = key_length
  
  FOR k = k1 TO 1 STEP -1
    mt(i) = (mt(i) Xor ((mt(i-1) Xor (mt(i-1) SHR 30)) * 1664525UL)) + InitKey(j) + j ' non linear 
    mt(i) = mt(i) AND &HffffffffUL  ' for WORDSIZE > 32 machines
    i = i + 1
    j = j + 1
    IF i >= NN THEN
      mt(0) = mt(NN-1)
      i = 1
    END IF
    IF j >= key_length THEN j = 0
  NEXT

  FOR k = NN-1 TO 1 STEP -1
    mt(i) = (mt(i) Xor ((mt(i-1) Xor (mt(i-1) SHR 30)) * 1566083941UL)) - i ' non linear 
    mt(i) = mt(i) AND &HffffffffUL ' for WORDSIZE > 32 machines
    i = i + 1
    IF i >= NN THEN
      mt(0) = mt(NN-1)
      i = 1
    END IF
  NEXT

  mt(0) = &H80000000UL ' MSB is 1; assuring non-zero initial array 
END SUB

FUNCTION IRanGen() AS UINTEGER
' Generates a Random number On [0,4294967295]-interval 

  DIM AS UINTEGER k, y

  ' generate NN words at one Time
  IF mti >= NN THEN
    ' If IRanMT() has not been called, a default initial seed is used 
    IF mti = NN + 1 THEN InitMT(5489UL)

    FOR k = 0 TO (NN-MM)-1
      y = (mt(k) AND UPPER_MASK) OR (mt(k+1) AND LOWER_MASK)
      mt(k) = mt(k+MM) XOR (y SHR 1) XOR mag01(y AND &H1UL)
    NEXT

    FOR k = (NN-MM) TO (NN-2)
      y = (mt(k) AND UPPER_MASK)OR (mt(k+1) AND LOWER_MASK)
      mt(k) = mt(k + (MM-NN)) XOR (y SHR 1) XOR mag01(y AND &H1UL)
    NEXT

    y = (mt(NN-1) AND UPPER_MASK) OR (mt(0) AND LOWER_MASK)
    mt(NN-1) = mt(MM-1) XOR (y SHR 1) XOR mag01(y AND &H1UL)

    mti = 0
  END IF

  y = mt(mti)
  mti = mti + 1

  ' Tempering 
  y = y XOR (y SHR 11)
  y = y XOR ((y SHL 7) AND &H9d2c5680UL)
  y = y XOR ((y SHL 15) AND &Hefc60000UL)
  y = y XOR (y SHR 18)

  FUNCTION = y
END FUNCTION

' ******************************************************************

' ------------------------------------------------------------------
' Global constants
' ------------------------------------------------------------------

CONST Z0 = 1.0# / 4294967296.0#        ' 1 / 2^32
CONST Z1 = 1.0# / 4294967295.0#        ' 1 / (2^32 - 1)
CONST Z2 = 1.0# / 9007199254740992.0#

' ******************************************************************

FUNCTION IRanGen31() AS UINTEGER
' 31-bit random integer

  RETURN CLNG(IRanGen() SHR 1)
END FUNCTION

FUNCTION RanGen1() AS DOUBLE
' 32-bit random real in [0,1]

  RETURN IRanGen() * Z1
END FUNCTION

FUNCTION RanGen2() AS DOUBLE
' 32-bit random real in [0,1)

  RETURN IRanGen() * Z0
END FUNCTION

FUNCTION RanGen3() AS DOUBLE
' 32-bit random real in (0,1)

  RETURN (CDBL(IRanGen()) + 0.5#) * Z0
END FUNCTION

FUNCTION RanGen53() AS DOUBLE
' 53-bit random real in [0,1)

  DIM A AS UINTEGER = IRanGen() SHR 5
  DIM B AS UINTEGER = IRanGen() SHR 6

  RETURN (A * 67108864.0# + B) * Z2
END FUNCTION
