' ******************************************************************
' BIGINT.BAS : Big Integer library for FBPano
' Interface for S. J. Schaper's LargeInt library
' ******************************************************************

#include "largeint.bas"

' ------------------------------------------------------------------
' Initialization of the library
' ------------------------------------------------------------------

function biginit (n as long) as long
' Allocates memory for n big integers
' Returns the max. number of digits

  if n <= 0 or n > 19450 then return 0

  biginit = int(4.51 * largeinit(n, ""))

  ' Delete file PrimFlgs.bin created by largeinit
  close : kill "PrimFlgs.bin"

  ' Store primality info for numbers up to ~ 10000
  primflags 10000
end function

' ------------------------------------------------------------------
' Writing and reading big integers
' (i) denotes the big integer of index i
' ------------------------------------------------------------------

sub setnum (i as long, n as long)
' Assigns the integer n to (i)

  letf i, n
end sub

sub setstr (i as long, s as string)
' Assigns the value of the string s to (i)

  readst i, s
end sub

sub setrnd (i as long, nbits as long)
' Assigns a nbit random integer to (i)
' The random integer is in [2^(n - 1) .. 2^n - 1]

  rndf i, nbits
end sub

function getstr (i as long) as string
' Returns the value of (i) in a character string

  dim a as string

  a = string(bufl(i), "0")
  cnvst a, i
  getstr = a
end function

' ------------------------------------------------------------------
' Copy, comparison, exchange
' ------------------------------------------------------------------

sub bigdup (i as long, j as long)
' (i) -> (j)

  dup i, j
end sub

sub bigswap (i as long, j as long)
' (i) <-> (j)

  swp i, j
end sub

function bigcmp (i as long, j as long) as long
' Compares (i) and (j)
' Returns -1 if (i) < (j), 0 if (i) = (j), 1 if (i) > (j)

  bigcmp = cmp(i, j)
end function

function bigisf (i as long, n as long) as long
' Returns -1 if (i) = n, 0 otherwise
' n is a signed 16-bit integer

  bigisf = isf(i, n)
end function

' ------------------------------------------------------------------
' Arithmetic functions
' ------------------------------------------------------------------

sub biginc (i as long, n as long)
' (i) + n -> (i)
' n is a signed 16-bit integer

  inc i, n
end sub

sub bigadd (i as long, j as long, k as long)
' (i) + (j) -> (k)

  dup i, k  ' (i) -> (k)
  add k, j  ' (k) + (j) -> (k)
end sub

sub bigsub (i as long, j as long, k as long)
' (i) - (j) -> (k)

  dup  i, k  ' (i) -> (k)
  subt k, j  ' (k) - (j) -> (k)
end sub

sub bigmul (i as long, j as long, k as long)
' (i) * (j) -> (k)

  mult i, j, k
end sub

sub bigdiv (i as long, j as long, k as long, m as long)
' (i) \ (j) -> (k)
' (i) mod (j) -> (m)

  dup  i, m     ' (i) -> (m)
  divd m, j, k  ' (m) mod (j) -> (m), (m) \ (j) -> (k)
end sub

function bigrat (i as long, j as long, n as long) as string
' Rational approximation of (i) / (j) with up to n digits

  dim a as string

  a = string(n, "0")
  ratcnv a, i, j, 10
  bigrat = a
end function

sub bigchs (i as long)
' -(i) -> (i)

  chs i
end sub

sub bigsqr (i as long, k as long)
' (i)^2 -> (k)

  squ i, k
end sub

function bigsqrt (i as long, k as long) as long
' if (i) is a perfect square: returns 0 and sqrt((i)) -> (k)
' otherwise returns -1

  if issqr(i, k) then bigsqrt = 0 else bigsqrt = -1
end function

sub bigpow (i as long, n as long, k as long)
' (i)^n -> (k)

  dup i, k   ' (i) -> (k)
  powr k, n  ' (k)^n -> (k)
end sub

sub bigfact (n as long, k as long)
' n! -> (k)
' n is a positive 16-bit integer
' 19000! has 73048 digits, corresponding to the max. capacity
' for a single integer, obtained by BigInit(1)

  fctl k, n
end sub

sub bigshl (i as long, n as long, k as long)
' (i) shl n -> (k)
' n is a positive 16-bit integer

  dup i, k
  lsft k, n
end sub

sub bigshr (i as long, n as long, k as long)
' (i) shr n -> (k)
' n is a positive 16-bit integer

  dup i, k
  rsft k, n
end sub

sub biggcd (i as long, j as long, k as long)
' Greatest commun divisor of (i) and (j) -> (k)

  gcd i, j, k
end sub

sub biglcm (i as long, j as long, k as long)
' Least common multiple of (i) and (j) -> (k)

  lcm i, j, k
end sub

' ------------------------------------------------------------------
' Prime numbers
' ------------------------------------------------------------------

function bigprimlist (nmin as long, nmax as long) as long
' Creates the file PrimeList.txt containing a list of prime numbers
' between nmin and nmax. Returns the number of primes.

  if nmax > getlp() then primflags nmax

  dim table as long, cf as long, n as long, sw as short

  table = freefile
  n = 0
  sw = 0

  open "PrimList.txt" for output as #table

  do
    cf = nextprime(sw)
    if cf >= nmin and cf <= nmax then
      n = n + 1
      print #table, cf
    end if
  loop until  cf > nmax

  close #table
  bigprimlist = n
end function

function bigprimnext (i as long) as long
' Replaces (i) by the first prime number > (i), according to Miller-Rabin test
' Result: 1 if (i) is prime, -1 if probably prime, 0 if composite

  const w = 7  ' number of witnesses
  const m = 0  ' largeint pointer

  dim df as long, d as long, r as long

  if isf(i, 1) then
    letf i, 2
    bigprimnext = 1
    exit function
  end if

  if isf(i, 2) or isf(i, 3) then
    bigprimnext = 1
    exit function
  end if

  bigprimnext = ispprm(i, m, w)

  dup i, m
  d = 5
  df = 4
  r = divint(m, 6)
  if r < 2 then d = 1: df = 2
  inc i, d - r

  dup i, m
  d = divint(m, getmd())  ' m mod 10000

  do
    if ispprm(i, m, w) then
      exit function
    end if

    df = 6 - df
    d = d + df
    inc i, df
  loop until  false
end function

function bigprimdec (i as long) as string
' Integer factorization of (i) by trial division
' Should run for factors up to about 10^9
' Returns a character string, e. g. "2^3 * 3^4 * 5^2"
' Unidentified factors noted as (?)

  const lim = 1000000007  ' search limit

  const dr = 0, q = 1, ii = 2, prod = 3  ' largeint pointers

  dim pr as long, sq as long, g as string, k as short, t as short

  dup i, ii     ' save number
  letf prod, 1  ' initialize product

  pr = 0: sq = 0
  if getl(i) > 2 then     ' preliminary tests
    sq = issqr(i, q)
    if sq then swp i, q   ' a is square
    pr = ispprm(i, q, 5)   ' a or sqrt(a) is pprime
  end if

  if pr then
    primstr i, 1, sq, g, prod
  else
    triald q, i, lim: t = 0
    while exq(dr, k, q, t)
      primstr dr, k, sq, g, prod
    wend
    if not isf(i, 1) then
      if ispprm(i, q, -5) then
        primstr i, 1, sq, g, prod   ' prime cofactor
      else
        primstr -1, 1, sq, g, prod  ' not found
      end if
    end if
  end if

  ' Restore number
  dup ii, i

  ' Check product of factors
  if cmp(i, prod) = 0 then
    bigprimdec = left(g, len(g) - 2)
  else
    bigprimdec = "(?)"
  end if
end function
