﻿' ****************************************
' Amstrad CPC graphic functions
' ****************************************

#include "FBImage.bi"

#include "chars.dat"

#include "colors.dat"

#ifndef NULL
  #define NULL cptr(any ptr, 0)
#endif

#define _2P32 &h100000000  ' 2^32 

' ----------------------------------------
' Global variables
' ----------------------------------------

dim shared as long  md            ' Graphic mode

dim shared as long  wgraf0, hgraf0' Screen size (by MODE)
dim shared as long  wgraf, hgraf  ' Screen size (by VIEW)
dim shared as long  xmin, ymax    ' Axis bounds

dim shared as long  xv1, yv1      ' Upper left corner for VIEW
dim shared as long  xv2, yv2      ' Lower right corner for VIEW

dim shared as ubyte wchar, hchar  ' Character size  
dim shared as ubyte w, h          ' Size of character element 

dim shared as long  lmax = 1000   ' Max number of rows
dim shared as long  cmax = 1000   ' Max number of columns    

dim shared as ulong fgcolor       ' Foreground color
dim shared as ulong bgcolor       ' Background color

dim shared as ulong ptr char(255) ' Characters

dim shared as long xc, yc         ' Current position in pixels
dim shared as long lc, cc         ' Current position (row, column)

dim shared as long blit           ' Blitting mode

dim shared as boolean ctrl_chars  ' For using control characters

' ----------------------------------------
' Plotting characters
' ----------------------------------------

sub plot_rect (img as ulong ptr, x as ubyte, y as ubyte, col as ulong)
' Plots a character element

  if w = 1 and h = 1 then
    pset img, (x, y), col
  elseif h = 1 then
    line img, (x, y)-(x + w - 1, y), col
  elseif w = 1 then
    line img, (x, y)-(x, y + h - 1), col  
  else 
    line img, (x, y)-(x + w - 1, y + h - 1), col, bf  
  end if
end sub

sub plot_line (img as ulong ptr, nline as ubyte, a as ubyte, s as string)
' Plots one line of a character
' Color coding by binary number (a) or string (s)

  dim as ubyte i, x, y
  dim as ulong col
  
  x = 0
  y = nline * h
  for i = 0 to 7
    if s <> "" then 
      col = iif(s[i] = 32, bgcolor, Colors(s[i]))
    else
      col = iif(bit(a, 7 - i), fgcolor, bgcolor)
    end if  
    plot_rect img, x, y, col  
    x += w
  next i  
end sub

sub create_char (i as ubyte)
' Creates character of ASCII code i as a sprite    

  if charcols(i) = "*" then      ' Character created by PICTOCHAR
    exit sub
  elseif charcols(i) = "" then
    for j as long = 0 to 7
      plot_line char(i), j, chars(i,j), ""
    next j
  else                           ' Character created by SYMBCOL
    for j as long = 0 to 7
      plot_line char(i), j, 0, mid(charcols(i), 8 * j + 1, 8)
    next j
  end if      
end sub
           
' ----------------------------------------
' Redefinition of characters
' ----------------------------------------
  
sub symbol (code as long, n0 as long, n1 as long, n2 as long, n3 as long, n4 as long, n5 as long, n6 as long, n7 as long) 

  if code < 0 or code > 255 then exit sub
  
  if n0 >= 0 and n0 <= 255 then chars(code,0) = cubyte(n0)
  if n1 >= 0 and n1 <= 255 then chars(code,1) = cubyte(n1)
  if n2 >= 0 and n2 <= 255 then chars(code,2) = cubyte(n2)
  if n3 >= 0 and n3 <= 255 then chars(code,3) = cubyte(n3)
  if n4 >= 0 and n4 <= 255 then chars(code,4) = cubyte(n4)
  if n5 >= 0 and n5 <= 255 then chars(code,5) = cubyte(n5)
  if n6 >= 0 and n6 <= 255 then chars(code,6) = cubyte(n6)
  if n7 >= 0 and n7 <= 255 then chars(code,7) = cubyte(n7)
end sub  

sub symbcol (code as long, s as string) 
  if code >= 0 and code <= 255 then charcols(code) = s + space(64 - len(s))
end sub

sub pictochar (filename as string, code as long) 
' Splits an image as a sequence of graphic characters
' code = index of first character

  dim as long w0, h0, w1, h1, ncol, nrow, x, y, x1, y1
  
  var img = LoadRGBAFile(filename) 
  if img = NULL then exit sub
  
  imageinfo img, w0, h0
  
  ncol = w0 \ wchar - 1  ' 0..ncol 
  nrow = h0 \ hchar - 1  ' 0..nrow
  
  if ncol < 0 or nrow < 0 then exit sub
  
  w1 = wchar - 1
  h1 = hchar - 1
 
  for y = 0 to nrow * hchar step hchar
    y1 = y + h1  
    for x = 0 to ncol * wchar step wchar  
      x1 = x + w1
      get img, (x, y)-(x1, y1), char(code)
      charcols(code) = "*"
      code +=1 : if code > 255 then exit sub
    next x  
  next y  
end sub

' ----------------------------------------
' Initialization of graphic mode
' ----------------------------------------

sub grafmode (wg as long = 640, hg as long = 400, we as long = 2, he as long = 2)

  screenres wg, hg, 32, 2, 4  ' 4 = (GFX_WINDOWED or GFX_NO_SWITCH)
  width wg \ 8, hg \ 16       ' Character size for the FB font 
  
  wgraf0 = wg
  hgraf0 = hg

  wgraf = wg
  hgraf = hg
  
  xv1 = 0
  yv1 = 0
  xv2 = wg - 1
  yv2 = hg - 1
  
  xmin = 0
  ymax = hgraf - 1

  w = we : wchar = 8 * w
  h = he : hchar = 8 * h
  
  cmax = wgraf \ wchar
  lmax = hgraf \ hchar

  lc = 1 : cc = 1
  
  dim as long i
  for i = 0 to 255
    if char(i) <> NULL then imagedestroy char(i)  
    char(i) = imagecreate(wchar, hchar, 0, 32)
  next i  
end sub  

' ----------------------------------------
' Graphic functions
' ----------------------------------------

sub mode (m as long, title as string = "", wg as long = 640, hg as long = 400, we as long = 2, he as long = 2) 
  
  md = m
  select case md
    case 0    : grafmode 640, 400, 4, 2
    case 1    : grafmode 640, 400, 2, 2
    case 2    : grafmode 640, 400, 1, 2
    case else : grafmode wg, hg, we, he
  end select  

  windowtitle title

  fgcolor = &hFFFFFF00
  bgcolor = &hFF000080

  color fgcolor, bgcolor
  cls
  
  ctrl_chars = TRUE
 end sub

sub origin (x as long = 0, y as long = 0, gauche as long = 0, droite as long = 0, haut as long = 0, bas as long = 0, fill_color as ulongint = _2P32, border_color as ulongint = _2P32) 

  ' Restore initial conditions before creating a new window 

  wgraf = wgraf0
  hgraf = hgraf0
  xmin = 0
  ymax = hgraf - 1
  xv1 = 0
  yv1 = 0
  xv2 = wgraf - 1
  yv2 = hgraf - 1
  
  if x > 0 and x <= wgraf then xmin = -x
  if y > 0 and y <= hgraf then ymax = hgraf - y 
  
  if gauche >= 0 and droite > gauche then 
    xv1 = gauche
    xv2 = droite
    wgraf = xv2 - xv1
    xmin += xv1
  end if  
      
  if bas >= 0 and haut > bas then 
    yv1 = hgraf - haut
    yv2 = hgraf - bas
    hgraf = yv2 - yv1
    ymax -= yv1
  end if    
  
  cmax = wgraf \ wchar
  lmax = hgraf \ hchar

  lc = 1 : cc = 1

  if fill_color = _2P32 and border_color = _2P32 then
    view (xv1, yv1)-(xv2, yv2)
  elseif border_color = _2P32 then
    view (xv1, yv1)-(xv2, yv2), fill_color
  elseif fill_color = _2P32 then
    view (xv1, yv1)-(xv2, yv2), , border_color
  else
    view (xv1, yv1)-(xv2, yv2), fill_color, border_color
  end if

  if fill_color <> _2P32 then bgcolor = fill_color : color , bgcolor
end sub

#define plotcolor iif(col = _2P32, fgcolor, col)

sub move (x as long, y as long) 
  xc = x : yc = y
end sub

function test (x as long, y as long) as ulong 
  xc = x : yc = y
  return point(x - xmin, ymax - y)
end function

sub plot (x as long, y as long, col as ulongint = _2P32) 
  pset (x - xmin, ymax - y), plotcolor
  xc = x : yc = y
end sub

sub ldraw (x as long, y as long, col as ulongint = _2P32) 
  line (xc - xmin, ymax - yc)-(x - xmin, ymax - y), plotcolor
  xc = x : yc = y
end sub

sub mover (x as long, y as long) 
  xc = xc + x : yc = yc + y
end sub

function testr (x as long, y as long) as ulong 
  xc = xc + x : yc = yc + y
  return point(xc - xmin, ymax - yc)
end function

sub plotr (x as long, y as long, col as ulongint = _2P32) 
  xc = xc + x : yc = yc + y
  pset (xc - xmin, ymax - yc), plotcolor
end sub

sub ldrawr (x as long, y as long, col as ulongint = _2P32) 
  line (xc - xmin, ymax - yc)-(xc - xmin + x, ymax - yc - y), plotcolor
  xc = xc + x : yc = yc + y
end sub

sub gcls () 
  cls
  lc = 1
  cc = 1
end sub

sub crlocate (x as long, y as long) 
  if x >= 1 and x <= cmax then cc = x
  if y >= 1 and y <= lmax then lc = y
  locate lc, cc
end sub

sub aprint (txt as string, x as ulongint = _2P32, y as ulongint = _2P32) 
' ---------------------------------------------------------------------------------
' Writes a string in graphic mode, using the Amstrad font
' Characters ASCII 0 to 31 may be interpreted as control characters

'        aprint chr(0) Takes control characters (default)
'        aprint chr(1) Print control characters
'
'        aprint chr(7)      Ring the bell (equivalent to BEEP)
'        aprint chr(8)      Shift cursor to the left
'        aprint chr(9)      Shift cursor to the right
'        aprint chr(10)     Move cursor down one line
'        aprint chr(11)     Move cursor up one line
'        aprint chr(12)     Clear the screen
'        aprint chr(13)     Return to line
'        aprint chr(14, n%) Equivalent of PAPER with n = colour index
'        aprint chr(15, n%) Equivalent of PEN with n = colour index
'
'        aprint chr(17) Erase the line until the last character
'        aprint chr(18) Erase the line from the last character
'        aprint chr(19) Erase from the screen top to the last character
'        aprint chr(20) Clear from the last character to the screen bottom
'
'        aprint chr$(22, n%) Set interaction with background ("bit blit"):
'
'          n% = 0 ==> PSET mode (normal)
'          n% = 1 ==> XOR mode
'          n% = 2 ==> AND mode
'          n% = 3 ==> OR mode
'          n% = 4 ==> ALPHA mode, with transparency = A of RGBA
'          n% = 5 ==> TRANS mode, with transparency = &hFF00FF (magenta)
'
'        aprint chr(30)       Equivalent of CRLOCATE 1, 1
'        aprint chr(31,x%,y%) Equivalent of CRLOCATE x%, y%
' ---------------------------------------------------------------------------------

  #define rect(x1, y1, x2, y2)  line (x1,y1)-(x2,y2), bgcolor, bf
  #define newpos(x, dx)         i += 1 : x = (txt[i] - 1) * dx
  
  ' Position in pixels
  var xx = iif(x = _2P32, (cc - 1) * wchar, x - xmin)
  var yy = iif(y = _2P32, (lc - 1) * hchar, ymax - y)

  var i = 0
  var imax = len(txt) - 1
  var code = 0
  
  while i <= imax
    code = txt[i]
    
    if cbool(code < 32) and ctrl_chars then
      
      select case code
        case 0  : ctrl_chars = TRUE
        case 1  : ctrl_chars = FALSE
        case 7  : beep
        case 8  : xx -= wchar
        case 9  : xx += wchar
        case 10 : yy += hchar
        case 11 : yy -= hchar
        case 12 : cls : xx = 0 : yy = 0
        case 13 : yy += hchar : xx = 0
        case 14 : i += 1 : bgcolor = Colors(txt[i] mod 256) : color , bgcolor
        case 15 : i += 1 : fgcolor = Colors(txt[i] mod 256) : color fgcolor
        case 17 : rect(0, yy, xx, yy + hchar)
        case 18 : rect(xx, yy, wgraf, yy + hchar)
        case 19 : rect(0, 0, wgraf, yy) : rect(0, yy, xx, yy + hchar)
        case 20 : rect(xx, yy, wgraf, yy + hchar) : rect(0, yy + hchar, wgraf, hgraf)
        case 22 : i += 1 : blit = txt[i] mod 6
        case 24 : swap fgcolor, bgcolor : color fgcolor, bgcolor
        case 30 : xx = 0 : yy = 0
        case 31 : newpos(xx, wchar) : newpos(yy, hchar)
      end select
      
    else
    
      create_char code
    
      select case blit
        case 0 : put (xx, yy), char(code), pset
        case 1 : put (xx, yy), char(code), xor
        case 2 : put (xx, yy), char(code), and
        case 3 : put (xx, yy), char(code), or
        case 4 : put (xx, yy), char(code), alpha, fgcolor shr 24
        case 5 : put (xx, yy), char(code), trans
      end select
    
      xx += wchar
    end if
      
    i += 1
  wend
end sub  

sub fbprint (txt as string, x as ulongint, y as ulongint) 
' Writes a string in graphic mode, using the FB font

  draw string (x - xmin, ymax - y), txt
end sub
  
sub paper (col as ulong) 
  bgcolor = col
  color , bgcolor
end sub
  
sub pen (col as ulong) 
  fgcolor = col
  color fgcolor
end sub

sub ink (i as ubyte, col as ulong) 
  Colors(i) = col
end sub

function get_ink (i as ubyte) as ulong 
  return Colors(i)
end function

sub rectangle (x as long, y as long, w as long, h as long) 
  line (x - xmin, ymax - y)-(x + w - xmin, ymax - y + h), fgcolor, b
end sub

sub rectangle_fill (x as long, y as long, w as long, h as long) 
  line (x - xmin, ymax - y)-(x + w - xmin, ymax - y + h), fgcolor, bf
end sub

sub arc (ctrx as long, ctry as long, rx as long, ry as long = 0, angle1 as long = 0, angle2 as long = 360) 
  var rmax = iif(rx >= ry, rx, ry)
  var a1 = iif(angle1 = 0, angle1, angle1 * 0.01745329252)
  var a2 = iif(angle2 = 360, 6.28, angle2 * 0.01745329252)
  var aspect = iif(ry = 0, 1, ry / rx)
  circle (ctrx - xmin, ymax - ctry), rmax, fgcolor, a1, a2, aspect
end sub

sub pie (ctrx as long, ctry as long, rx as long, ry as long = 0) 
  var rmax = iif(rx >= ry, rx, ry)
  var aspect = iif(ry = 0, 1, ry / rx)
  circle (ctrx - xmin, ymax - ctry), rmax, fgcolor, , , aspect, f
end sub

sub fill (paint_col as ulongint, border_col as ulongint = _2P32) 
  if border_col = _2P32 then border_col= fgcolor
  paint (xc - xmin, ymax - yc), paint_col, border_col
end sub

sub fill_pattern (numchar as long, border_col as ulongint = _2P32) 
  var s = charcols(numchar)
  dim as string pat
  dim as long i
  for i = 0 to 63
    pat += MKL(cuint(Colors(s[i])))
  next i
  if border_col = _2P32 then border_col= fgcolor
  paint (xc - xmin, ymax - yc), pat, border_col
end sub

' ----------------------------------------
' Mouse
' ----------------------------------------

sub get_mouse (byref x as long, byref y as long, byref btn as long, byref wheel as long) 
  dim as long x1, y1
  getmouse x1, y1, wheel, btn
  x = x1 + xmin - xv1
  y = ymax - y1 + yv1
end sub

' ----------------------------------------
' Graphic files
' ----------------------------------------

sub img_save (filename as string, transp as long = FALSE) 
  
  dim as string  ext
  dim as any ptr img
  
  var fname = trim(filename)
  var i     = instr(fname, ".")
  
  if i = 0 then 
    fname += ".bmp"
    ext = "bmp"
  else     
    ext = lcase(mid(fname, i + 1))
  end if
  
  img = imagecreate(xv2 - xv1 + 1, yv2 - yv1 + 1)
  get (0, 0)-(xv2 - xv1, yv2 - yv1), img

  select case ext
    case "bmp"
      bsave fname, img
    case "png"
      SavePNGFile img, fname, transp
  end select

  imagedestroy img
end sub
 
sub img_load (filename as string) 
  
  var fname = trim(filename)
  if instr(fname, ".") = 0 then fname += ".bmp"

  var img = LoadRGBAFile(fname)
  if img = NULL then exit sub

  put (0, 0), img, PSET

  imagedestroy img
end sub

sub img_load_trans (filename as string) 
  
  dim as string       fname = filename
  dim as any ptr      pixdata
  dim as integer      col, w, h, pitch, x, y
  dim as uinteger ptr pixel

  if instr(fname, ".") = 0 then fname += ".bmp"

  var img = LoadRGBAFile(fname)
  if img = NULL then exit sub

  imageinfo img, w, h, , pitch, pixdata

  pixel = pixdata
  col = pixel[0]

  for y = 0 to h - 1
    pixel = pixdata + y * pitch
    for x = 0 to w - 1
      if pixel[x] = col then pixel[x] = 0
    next x
  next y

  put (0, 0), img, alpha

  imagedestroy img
end sub

' *******************************************************************
' HSV / RGB conversion
' *******************************************************************
' Adapted from http://www.cs.rit.edu/~ncs/color/t_convert.html
' R, G, B values are from 0 to 255
' H = [0..360], S = [0..1], V = [0..1]
' if S = 0, then H = -1 (undefined)
' *******************************************************************

dim shared as double RR, GG, BB

sub RGBtoHSV (R as long, G as long, B as long, byref H as double, byref S as double, byref V as double) 
' Convert RGB to HSV

  RR = R / 255
  GG = G / 255
  BB = B / 255

  dim as double Min, Max, Delta

  Min = RR
  if GG < Min then Min = GG
  if BB < Min then Min = BB

  Max = RR
  if GG > Max then Max = GG
  if BB > Max then Max = BB

  V = Max

  Delta = Max - Min

  if Max <> 0 then
    S = Delta / Max
  else
    S = 0    ' R = G = B = 0  ==> S = 0, V is undefined
    H = -1
  end if

  if RR = Max then
    H = ( GG - BB ) / Delta      ' between yellow & magenta
  elseif GG = Max then
    H = 2 + ( BB - RR ) / Delta  ' between cyan & yellow
  else
    H = 4 + ( RR - GG ) / Delta  ' between magenta & cyan
  end if

  H = H * 60                     ' degrees
  if H < 0 then H = H + 360
end sub

sub HSVtoRGB alias "HSVtoRGB" (H as double, S as double, V as double, byref R as long, byref G as long, byref B as long) 
' Convert HSV to RGB

  if S = 0 then  ' achromatic (grey)
    R = V * 255
    G = R
    B = R
    exit sub
  end if

  dim as integer I
  dim as double  Z, F, P, Q, T

  Z = H / 60     ' sector 0 to 5
  I = int(Z)
  F = frac(Z)
  P = V * (1 - S)
  Q = V * (1 - S * F)
  T = V * (1 - S * (1 - F))

  select case I
    case 0
      RR = V
      GG = T
      BB = P
    case 1
      RR = Q
      GG = V
      BB = P
    case 2
      RR = P
      GG = V
      BB = T
    case 3
      RR = P
      GG = Q
      BB = V
    case 4
      RR = T
      GG = P
      BB = V
    case 5
      RR = V
      GG = P
      BB = Q
  end select

  R = RR * 255
  G = GG * 255
  B = BB * 255
end sub

' ----------------------------------------
' Alphabetical INKEY
' ----------------------------------------

function ainkey () as string 
  
  dim as string key = inkey()

  if key = "" then return key
  
  dim as long code   = 0
  dim as long nchars = len(key)
  
  code = asc(right(key, 1))
  if nchars > 1 then code += 256
  
  select case code
    case 1    : return "CTRL+A"
    case 2    : return "CTRL+B"
    case 3    : return "CTRL+C"
    case 4    : return "CTRL+D"
    case 5    : return "CTRL+E"   
    case 6    : return "CTRL+F"   
    case 8    : return "BACKSPACE"
    case 9    : return "TAB"      
    case 10   : return "CTRL+J"   
    case 11   : return "CTRL+K"   
    case 12   : return "CTRL+L"
    case 13   : return "ENTER"
    case 14   : return "CTRL+N"
    case 15   : return "CTRL+O"
    case 16   : return "CTRL+P"
    case 17   : return "CTRL+Q"
    case 18   : return "CTRL+R"
    case 19   : return "CTRL+S"
    case 20   : return "CTRL+T"
    case 21   : return "CTRL+U"
    case 22   : return "CTRL+V"
    case 23   : return "CTRL+W"
    case 24   : return "CTRL+X"
    case 25   : return "CTRL+Y"
    case 26   : return "CTRL+Z"
    case 27   : return "ESCAPE"
    
    case 32 to 126 : return key
    
    case 315  : return "F1"
    case 316  : return "F2"
    case 317  : return "F3"
    case 318  : return "F4"
    case 319  : return "F5"
    case 320  : return "F6"
    case 321  : return "F7"
    case 322  : return "F8"
    case 323  : return "F9"
    case 324  : return "F10"
    case 327  : return "HOME"
    case 328  : return "UP"
    case 329  : return "PAGEUP"
    case 331  : return "LEFT"
    case 333  : return "RIGHT"
    case 335  : return "END"
    case 336  : return "DOWN"
    case 337  : return "PAGEDOWN"
    case 338  : return "INSERT"
    case 339  : return "DELETE"
  end select
end function



