' *******************************************************
' Maze exploration. 
' The sprites are taken from a FreeBASIC game:
' https://sourceforge.net/projects/mydungeon-chung/
' *******************************************************

#include "amsgraph.bi"

declare sub GenMaze (maze() as string)
declare sub PrintLegend ()
declare sub DrawMaze (maze() as string, wall as string, warrior as string, door as string, xp as long, yp as long)
declare sub ExploreMaze (maze() as string, wall as string, warrior as string, door as string)

const W = 30, H = 20  ' Maze dimensions (must be even)

dim as string maze(W, H), wall, warrior, door, key

mode 3, "Maze exploration", 32 * (W + 1), 32 * (H + 2), 4, 4

paper CL_BLACK : gcls

' Convert spritesheet to characters, starting at ASCII 128
pictochar "maze.png", 128

warrior = chr(128)
wall = chr(129)
door = chr(130)

Start:

gcls
GenMaze maze()
DrawMaze maze(), wall, warrior, door, 2, 2
PrintLegend 
ExploreMaze maze(), wall, warrior, door

do
  key = ainkey()
loop until key <> ""

if key <> "ESCAPE" then goto Start

' *******************************************************

sub GenMaze(maze() as string)
' ---------------------------------------------------
' Maze generation
' Adapted from QuickBasic
' https://rosettacode.org/wiki/Maze_generation#BASIC
' ---------------------------------------------------

  dim as long x, y
  dim currentx as long, currenty as long, oldx as long, oldy as long
  dim done as long, i as long

  for x = 0 to W
    for y = 0 to H
      maze(x, y) = "#"
    next y
  next x
 
  randomize timer
 
  ' initial start location
  currentx = int(rnd * (W - 1))
  currenty = int(rnd * (H - 1))

  ' value must be odd
  if currentx mod 2 = 0 then currentx = currentx + 1
  if currenty mod 2 = 0 then currenty = currenty + 1

  maze(currentx, currenty) = " "
 
  ' generate maze

  done = 0
  while done = 0
    for i = 0 to 99
      oldx = currentx
      oldy = currenty
 
      ' move in random direction
      select case int(rnd * 4)
        case 0
          if currentx + 2 < W then currentx = currentx + 2
        case 1
          if currenty + 2 < H then currenty = currenty + 2
        case 2
          if currentx - 2 > 0 then currentx = currentx - 2
        case 3
          if currenty - 2 > 0 then currenty = currenty - 2
      end select
 
      ' if cell is unvisited then connect it
      if maze(currentx, currenty) = "#" then
        maze(currentx, currenty) = " "
        maze((currentx + oldx) \ 2, (currenty + oldy) \ 2) = " "
      end if
    next i
 
    ' check if all cells are visited
    done = 1
    for x = 1 to W - 1 step 2
      for y = 1 to H - 1 step 2
        if maze(x, y) = "#" then done = 0
      next y
    next x
  wend
end sub

sub PrintLegend ()

  const TXT = "Guide the warrior to the exit with the arrow keys"

  pen CL_BRIGHT_WHITE : plot 0, 30 : ldrawr 32 * (W + 1), 0 

  pen CL_BRIGHT_GREEN : fbprint TXT, 16 * (W + 1) - 4 * len(TXT), 20
end sub

sub DrawMaze(maze() as string, wall as string, warrior as string, door as string, xp as long, yp as long)
' xp%, yp% : Position du personnage 

  dim x as long, y as long, x1 as long, y1 as long, dx as long, dy as long, dist as long, op as long

  aprint chr(22, 4)  ' Activates transparent mode

  for y = 0 to H
    for x = 0 to W
      x1 = x + 1
      y1 = y + 1
      dx = x1 - xp
      dy = y1 - yp
      dist = sqr(dx * dx + dy * dy)
      if dist < 5 then op = 250 - 50 * dist else op = 0
      pen rgba(0, 0, 0, op)  
      crlocate x1, y1
      if maze(x, y) = "#" then aprint wall else aprint " "
    next x
  next y
  
  pen CL_WHITE
  crlocate W, H : aprint door
  crlocate xp, yp : aprint warrior
end sub

sub ExploreMaze(maze() as string, wall as string, warrior as string, door as string)

  const TXT = "Type ESC to quit, another key to replay"

  dim x as long, y as long, x1 as long, y1 as long, a as string

  x = 2 : y = 2 : x1 = x : y1 = y 

  do
    a = ainkey 
    
    select case a
      case "RIGHT" : if x < W then x1 = x + 1
      case "LEFT"  : if x > 2 then x1 = x - 1
      case "DOWN"  : if y < H then y1 = y + 1
      case "UP"    : if y > 2 then y1 = y - 1
    end select

    if (x1 <> x or y1 <> y) and (maze(x1 - 1, y1 - 1) = " ") then 
      DrawMaze maze(), wall, warrior, door, x1, y1
      crlocate x, y : aprint " "
      crlocate x1, y1 : aprint warrior
      x = x1 : y = y1
    else
      x1 = x : y1 = y
    end if
  loop until (x = W and y = H)

  pen CL_BLACK : rectangle_fill 0, 20, 32 * (W + 1), 20 
  pen CL_BRIGHT_CYAN : fbprint TXT, 16 * (W + 1) - 4 * len(TXT), 20
end sub


